require 'aml'
require 'dialogs/team/wizard/team_step'

include System::IO

include Axure::Platform
include Axure::Platform::CommonDialogs

include Axure::Client::Common::Team
include Axure::Client::Dialogs::Infos
include Axure::Client::Operations
include Axure::Platform::Storage::RevisionTracking

class TeamSharedDirectoryWizardStep < TeamStep

	def initialize()
		super

		@show_next = true
		@show_previous = true

		@title = intl_str("Shared Project Directory")
		@description = intl_str("Select the directory where the Shared Project Directory will be created.")

		@path_verified = false
		@help_label.text = intl_str(String.Format("This directory is commonly on a network drive where others can access the Shared Project.\n"+
"Ex: {0}" + "Public"+SEP+"OurSharedDirectory\n\n"+
"The Shared Project Directory will be created in this directory with the project name.\n" +
"Ex: {0}" + "Public"+SEP+"OurSharedDirectory"+SEP+"ProjectName\n\n"+
"The Shared Directory can also be a URL for an SVN directory. An SVN server must already be configured.\nEx: http://svn.myserver.com/OurSharedDirectory\nEx: svn://www.myserver.com/svn/OurSharedDirectory.", AxPlatform.Features.SupportsUNCPath ? '\\\\' : '/Volumes/' ))
	end

	def browse_clicked
		open_dir_dlg = AxOpenFolderDialogInfo.new 
		open_dir_dlg.description = intl_str("Select the directory that will contain the Shared Project.")
		if open_dir_dlg.show_dialog() == AxDialogResult.OK
    	@shared_directory_textbox.text = open_dir_dlg.folder_name
		end
	end

	def shared_directory_text_changed
		@path_verified = false
		self.can_next = @shared_directory_textbox.text.strip().length > 0
	end

	def wizard_control=(value)
		super(value)
		if wizard_control && wizard_control.dialog_info
			@shared_directory_textbox.text = wizard_control.dialog_info.get_property(TEAM_SHARED_DIRECTORY_KEY)
		end
	end

	def save_step
		wizard_control.dialog_info.set_property(TEAM_SHARED_DIRECTORY_KEY, @shared_directory_textbox.text)
	end

	def verify
		return true if @path_verified

		checkPathOp = TeamProjectOperation.new
		checkPathOp.CheckVersion = false
		checkPathOp.OperationName = intl_str("Checking paths . . . ")
		checkPathOp.Work.add(lambda { 
			@path_verified = checkPathOp_Work()
		})

		progressDialog = ProgressDialogInfo.new()
		progressDialog.TrapErrors = false
		progressDialog.Operation = checkPathOp
		progressDialog.SupportsCancel = false
		progressDialog.MarqueeStyle = true
		progressDialog.ShowDialog()

		return @path_verified
	end

	def checkPathOp_Work
		begin
			repopath = AxUri.new(@shared_directory_textbox.text)
		rescue StandardError, System::Exception 
			AxPlatform.ShowMessageBox(@shared_directory_textbox.text + intl_str(" is not a valid path."))
			return false
		end

		userInput = wizard_control.dialog_info.get_property(TEAM_PROJECT_NAME_KEY)
		uriSep = System::String.new("/")

		#share project folder
		str = repopath.ToString() + uriSep + userInput
		projectpath = AxUri.new(str);

		#SHARE folder Check if directory exists. If not create it.
		createdPath = false;
		if (repopath.IsFile) 
			#creation only happens for file://
			if !Directory.Exists(repopath.LocalPath)

				if AxPlatform.ShowMessageBox(String.Format(
					intl_str("{0} does not exist.\nWould you like to create it?"), repopath), 
					intl_str("Alert"), AxMessageBoxButtons.YesNo) == AxDialogResult.Yes
					begin
						Directory.CreateDirectory(repopath.LocalPath);
						createdPath = true;
					rescue StandardError, System::Exception 
						AxPlatform.ShowMessageBox(
							String.Format(intl_str("{0} could not be created.\nPlease select another folder."), repopath))
							return false
					end #begin-rescue 
				else 
					return false
				end #MessageBox 
			else
				if TeamClientFunctions.CheckSharedFolderPath(repopath.LocalPath)
					AxPlatform.ShowMessageBox(
						String.Format(intl_str("{0} is in another shared project.\nPlease choose another path."), projectpath))
						return false
				end #CheckSharedFolderPath 
			end #!Directory.Exists 
		else 
			if TeamClientFunctions.CheckConnectionFromUri(repopath)
				if TeamClientFunctions.IsShareProject(repopath)
					AxPlatform.ShowMessageBox(String.Format(
						intl_str("{0} is in another shared project.\nPlease choose another path."),
						projectpath.AbsoluteUri))
						return false
				end #IsShareProject
			else 
				return false
			end #CheckConnectionnection
		end #repopath.IsFile


		return false if !CheckProjectPath(projectpath, createdPath) 

		return true
	end 


	def CheckProjectPath(projectpath, createdPath) 
		#check project path
		if projectpath.IsFile 
			if !createdPath && Directory.Exists(projectpath.LocalPath) 
				if Directory.GetDirectories(projectpath.LocalPath).Length != 0 || 
					Directory.GetFiles(projectpath.LocalPath).Length != 0 
					AxPlatform.ShowMessageBox(String.Format(
						intl_str("{0} is not empty.\nThe shared directory must be empty."), projectpath))
						return false
				end 
			else 
				begin 
					Directory.CreateDirectory(projectpath.LocalPath)
				rescue StandardError, System::Exception 
					AxPlatform.ShowMessageBox(String.Format(
						intl_str("{0} could not be created.\nPlease select another folder."), projectpath))
						return false
				end 
			end #Directory.Exists 
		else 
			return CheckRemoteProjectPath(projectpath);
		end #IsFile

		return true
	end #method 

	def CheckRemoteProjectPath(projectpath) 
		#we can connect to repopath
		repository = AxPlatform.PlatformServices.GetRevisionTrackingProvider()
		availability = repository.Availability(projectpath)
		case (availability) 
		when RevisionProviderState.Online
			#check if shareproject or empty
			ls = List.of(AxUri).new(repository.ListContents(projectpath))
			if ls.length != 0
				AxPlatform.ShowMessageBox(String.Format(
					intl_str("{0} is not empty.\nThe shared project location must be empty."),
					projectpath.AbsoluteUri));
				return false
			end 
		when RevisionProviderState.Offline
		when RevisionProviderState.PartiallyOnline
			#the project folder doesnt exist.  lets try to create it
			projectName = wizard_control.dialog_info.get_property(TEAM_PROJECT_NAME_KEY)
			begin 
				repository.MakeDirectory(projectpath.AbsoluteUri, "Create Shared Project folder " + projectName)
			rescue StandardError, System::Exception 
				AxPlatform.ShowMessageBox(String.Format(
					intl_str("{0} could not be created.\nPlease select another location."), projectpath.AbsoluteUri))
				return false
			end
		end #case 
		return true
	end
		 
end #class
