require 'aml'
require 'axure_enumerable'
require 'Model'
require 'interop'

include Axure::Platform
include Axure::Platform::Component::Commands
include Axure::Platform::Component::Menus
include Axure::Platform::Controls

include Axure::Client::Operations
include Axure::Client::Interface
include Axure::Client::Common::Team

include Axure::DocumentModel


class TeamProjectManager < AML::Control
	def initialize()
		self.auto_size = true
		initialize_component("dialogs/team/project_manager.aml")
		initialize_commands()
		initialize_menu()
	end

	def dialog_info=(_info)
		@client = _info["Client"];
    @dialog_info = _info

    _info.default_size = AxSizeF.new(600,400)
    _info.min_size = AxSizeF.new(400,300)
    _info.title = intl_str("Manage Shared Project")

    _info.before_shown.add(lambda { @package_list.set_widths_proportional })

    uri = @client.CurrentFile.Team.ProviderUri; 
    @shared_directory_label.text = (uri.IsFile || uri.IsUnc) ? uri.LocalPath : uri.ToString()
	end

	def initialize_commands
		checkInKeepLockStr =   "CheckInKeepLock";
		checkInReleaseLockStr =   "CheckInReleaseLock";
		teamCommandNames = ["Update", "CheckOut", checkInKeepLockStr, checkInReleaseLockStr, "UndoCheckOut"]

		teamHandlers = [ :team_Update, :team_Checkout, :team_CheckInKeepLock, :team_CheckInReleaseLock, :team_UndoCheckout]

		teamCommandInfos = [ButtonCommandInfo.new, ButtonCommandInfo.new, ButtonCommandInfo.new,
			ButtonCommandInfo.new, ButtonCommandInfo.new ]

		(0..teamHandlers.length - 1).each { | i |

			h  = (teamHandlers[i])
			teamCommandInfos[i].ValueChanged.add(lambda {
				self.send(h)
			} )
		}

		@updateCommand = Command.new(teamCommandNames[0], teamCommandInfos[0])
		@checkOutCommand = Command.new(teamCommandNames[1], teamCommandInfos[1])
		@checkInPackageKeepLockCommand = Command.new(teamCommandNames[2], teamCommandInfos[2])
		@checkInPackageReleaseLockCommand = Command.new(teamCommandNames[3], teamCommandInfos[3])
		@undoCheckOutCommand = Command.new(teamCommandNames[4], teamCommandInfos[4])
	end

	def initialize_menu

		@updatePackageMenuItem = MenuItem.new
		@updatePackageMenuItem.Text =   intl_str("&Get Changes")
		@updatePackageMenuItem.Command = @updateCommand
		@updatePackageMenuItem.IconList = ImageListContainer.TeamActionImageList
		@updatePackageMenuItem.IconIndex = 0

		@checkOutPackageMenuItem = MenuItem.new
		@checkOutPackageMenuItem.Text =   intl_str("Check &Out")
		@checkOutPackageMenuItem.Command = @checkOutCommand
		@checkOutPackageMenuItem.IconList = ImageListContainer.TeamActionImageList
		@checkOutPackageMenuItem.IconIndex = 1

		@checkInPackageKeepLockMenuItem = MenuItem.new
		@checkInPackageKeepLockMenuItem.Text =   intl_str("&Send Changes")
		@checkInPackageKeepLockMenuItem.Command = @checkInPackageKeepLockCommand
		@checkInPackageKeepLockMenuItem.IconList = ImageListContainer.TeamActionImageList
		@checkInPackageKeepLockMenuItem.IconIndex = -1

		@checkInPackageReleaseLockMenuItem = MenuItem.new
		@checkInPackageReleaseLockMenuItem.Text =   intl_str("Check &In")
		@checkInPackageReleaseLockMenuItem.Command = @checkInPackageReleaseLockCommand
		@checkInPackageReleaseLockMenuItem.IconList = ImageListContainer.TeamActionImageList
		@checkInPackageReleaseLockMenuItem.IconIndex = 2

		@undoCheckOutPackageMenuItem = MenuItem.new
		@undoCheckOutPackageMenuItem.Text = intl_str("&Undo Check Out")
		@undoCheckOutPackageMenuItem.Command = @undoCheckOutCommand
		@undoCheckOutPackageMenuItem.IconList = ImageListContainer.TeamActionImageList
		@undoCheckOutPackageMenuItem.IconIndex = -1

		@teamCtxMenu = ContextMenu.new
		@teamCtxMenu.Name =   "Team Menu"
		items = [@updatePackageMenuItem, @checkInPackageKeepLockMenuItem, @checkOutPackageMenuItem,
			@checkInPackageReleaseLockMenuItem, @undoCheckOutPackageMenuItem]
		@teamCtxMenu.MenuItems = items.to_clr_array_of_type(MenuItem)
	end

	def getSelectedPackageIds()
		return [@control_to_tinfo[@package_list.selected_item].PackageId].to_clr_list
	end

	def team_Update
		TeamClientFunctions.UpdatePackages(@client, getSelectedPackageIds())
		refresh_clicked
	end

	def team_Checkout
		TeamClientFunctions.CheckOutPackages(@client, getSelectedPackageIds())
		refresh_clicked
	end

	def team_CheckInKeepLock
		TeamClientFunctions.CheckInPackages(@client, getSelectedPackageIds(), false)
		refresh_clicked
	end

	def team_CheckInReleaseLock
		TeamClientFunctions.CheckInPackages(@client, getSelectedPackageIds(), true)
		refresh_clicked
	end

	def team_UndoCheckout
		TeamClientFunctions.UndoCheckOutPackages(@client, getSelectedPackageIds())
		refresh_clicked
	end


	def package_list_clicked(sender, e)
		if e.Button == AxMouseButtons.Right

			item = @package_list.selected_item
			tInfo = @control_to_tinfo[item]

      canBeEnabled = tInfo != nil # true for deleted from repo
			@updateCommand.Enabled = canBeEnabled
			@checkOutCommand.Enabled = canBeEnabled
			@checkInPackageKeepLockCommand.Enabled = canBeEnabled
			@checkInPackageReleaseLockCommand.Enabled = canBeEnabled
			@undoCheckOutCommand.Enabled = canBeEnabled

      if(canBeEnabled) 
        @updateCommand.Enabled &= tInfo.IsOutdated
        @checkOutCommand.Enabled &= tInfo.CanCheckOut
        @checkInPackageKeepLockCommand.Enabled &= tInfo.IsCheckOut && (item.get_control_at_index(5).Text == intl_str("Yes"))
        @checkInPackageReleaseLockCommand.Enabled &= tInfo.CanCheckIn
        @undoCheckOutCommand.Enabled &= tInfo.CanUndoCheckOut
      end

			@client.ShowContextMenu(@teamCtxMenu, item.PointToScreen(e.Location))
		end
	end

	def refresh_clicked
		teamCtrl = @client.CurrentFile.Team

		#begin
			return if !TeamClientFunctions.CheckConnection(@client)
			@client.CommitOuterObject();

			teamOp = TeamProjectOperation.new
			teamOp.CheckVersion = false
			teamOp.File = @client.CurrentFile
			teamOp.OperationName = intl_str("Getting Status")

			teamOp.Work.add(lambda {
				@localPackageTeamInfo, @newFromRepo, @deletedFromRepo, @deletedLocally = teamCtrl.GetSharedProjectStatus()
			})

			progressDialog = Axure::Client::Dialogs::Infos::ProgressDialogInfo.new
			progressDialog.Operation = teamOp
			progressDialog.SupportsCancel = false
			progressDialog.MarqueeStyle = true
			progressDialog.ShowDialog()

			refresh_package_list_view()
		#rescue
			#teamCtrl.EndOperationsOnError()
			#TeamClientFunctions.ShowOperationErrorDialog(intl_str("Get Shared Project Status"), $!)
		#end
	end

	def display_type(pInfo)
		case pInfo.PackageTypeName
		when AxureType.Page
			type = intl_str("Page")
		when AxureType.Master
			type = intl_str("Master")
		else
			type = intl_str("Property")
		end
	end

	def refresh_package_list_view
		@package_list.clear_controls()
		@control_to_tinfo = Hash.new


		delRepoSet = Set.new
		delLocalSet = Set.new

		@deletedFromRepo.each { | dInfo | delRepoSet.add(dInfo.PackageId) }
		@deletedLocally.each { | dInfo | delLocalSet.add(dInfo.PackageId) }

		#lvItem....AddRange(my status, shared status, get status, send status)
		@newFromRepo.each do | pInfo |
			@package_list.add_item(display_type(pInfo), pInfo.PackageName,
														 intl_str("N/A"), intl_str("New"), intl_str("Yes"), intl_str("No") )
		end

		@deletedFromRepo.each do | pInfo |
			items = [display_type(pInfo), pInfo.PackageName]
			team_info = @localPackageTeamInfo.find proc { | t | return t.PackageId == pInfo.PackageId }

			#lvItem....AddRange(my status, shared status, get status, send status)
			if delLocalSet.include?(pInfo.PackageId)
				items += [intl_str("Deleted"), intl_str("Deleted"), intl_str("Yes"), intl_str("No")]
			elsif team_info.IsCheckOut
				if (team_info.IsModified || @client.CurrentFile.DirtyPackages.Contains(team_info.PackageId))
					#if modififed in WC or temp dir
					items += [intl_str("Check Out"), intl_str("Deleted"), intl_str("No"), intl_str("Yes")]
				else
					items += [intl_str("Check Out"), intl_str("Deleted"), intl_str("Yes"), intl_str("No")]
				end
			elsif team_info.IsConflicted
				items += [intl_str("Conflicted"), intl_str("Deleted"), intl_str("No"), intl_str("Yes")]
			else
				items += [intl_str("No Change"), intl_str("Deleted"), intl_str("Yes"), intl_str("No")]
			end

			@package_list.add_item(items.to_clr_list)
		end

		@deletedLocally.each do | pInfo |
			next if delRepoSet.include?(pInfo.PackageId)

			items = [display_type(pInfo), pInfo.PackageName]

			team_info = @localPackageTeamInfo.find proc { | t | return t.PackageId == pInfo.PackageId }

			if team_info.IsExternalCheckOut
				items += [intl_str("Deleted"),
					System::String.Format(intl_str("Checked out by {0}"), team_info.PackageRepoInfo.LockInfo.Owner),
					intl_str("No"), intl_str("Yes") ]
			else
				items += [intl_str("Deleted"), intl_str("Available for check out"), intl_str("No"), intl_str("Yes")]
			end

			@package_list.add_item(items.to_clr_list)
		end

		@localPackageTeamInfo.each { | team_info |
			next if (team_info.PackageId == @client.Document.PackageId )

			next if delRepoSet.include?(team_info.PackageId)

			pInfo = @client.Document.GetPackageInfo(team_info.PackageId)

			#deleted pkg info.  need to find out where
			# the pkg originated from
			next if (pInfo == nil)

			items = [ display_type(pInfo), pInfo.PackageName ]

			#myStat
			#sharedStat

			#my status column
			if team_info.IsNew
				myStat = intl_str("New")
			elsif team_info.IsConflicted
				myStat =  System::String.Format(intl_str("Conflicted with {0}"), team_info.PackageRepoInfo.LastAuthor)
			elsif team_info.IsUnsafeCheckOut
				myStat = intl_str("Unsafely checked out")
			elsif team_info.IsCheckOut
				if (team_info.PackageRepoInfo.LockInfo.LockBroken ||
						team_info.PackageRepoInfo.LockInfo.LockStolen)
					myStat = intl_str("Check out stolen")
				else
					myStat = intl_str("Checked out")
				end
			else
				myStat = intl_str("No change")
			end #myStat

			#shared folder status column
			if team_info.IsNew
				sharedStat = intl_str("N/A")
			elsif team_info.PackageRepoInfo.LockInfo.LockBroken
				sharedStat = intl_str("Available for check out")
			elsif team_info.PackageRepoInfo.Locked
				sharedStat =  System::String.Format(intl_str("Checked out by {0}"), team_info.PackageRepoInfo.LockInfo.Owner)
			elsif !team_info.PackageRepoInfo.Locked
				sharedStat = intl_str("Available for check out")
			else
				#Debug.Fail(  "Not possible")
				sharedStat = intl_str("N/A")
			end #sharedStat

			items << myStat
			items << sharedStat

			#get change column
			if team_info.IsOutdated
				items << intl_str("Yes")
			else
				items << intl_str("No")
			end

			#send change column
			if (team_info.IsNew || team_info.IsModified ||
					@client.CurrentFile.DirtyPackages.Contains(team_info.PackageId))
				items << intl_str("Yes")
			else
				items << intl_str("No")
			end

			c = @package_list.add_item(items.to_clr_list)
			@control_to_tinfo[c] = team_info

		} #localPackageTeamInfo.each
	end #refresh_package_list_view
end #class
