require 'aml'
require 'interop'
require 'Core'

include System::IO
include Axure::IO

class RecoverFileDialog < AML::Control
  def initialize()
    initialize_component('dialogs/recover_file_dlg/recover_file_dlg.aml')
  end

  def dialog_info=(val)
    @info = val
    @info.before_close.add(lambda { | s, e| before_close(s, e) } )
    @info.before_shown.add(lambda { @files.set_widths_proportional })
    @info.default_size = AxSizeF.new(640,300)
    @info.min_size = AxSizeF.new(400,250)
    @files.set_column_comparer(0, SortTypes.date_as_string)
    @control_to_filename = Hash.new
    @filename_to_control = Hash.new
    populate_recent_files
  end

  def days_changed
    populate_recent_files
  end

  def file_selected(sender, e)
    e.Handled = true
    recover_file
  end

  def populate_recent_files
    selected_filename = @control_to_filename[@files.selected_item]
    @files.suspend_layout
    @files.clear_controls
    @files.selected_index = -1

    directory_info = System::IO::DirectoryInfo.new(AxPlatform.platform_services.application_files.app_temporary_path)

    @control_to_filename.clear
    @filename_to_control.clear
    populate_files_by_type(GetFiles.get_files_with_extension(directory_info, ".rp", false), /(.+) - ([0-9]+).rp/, ".rp")
    populate_files_by_type(GetFiles.get_files_with_extension(directory_info, ".rplib", false), /(.+) - ([0-9]+).rplib/, ".rplib")

    @files.sort_by_column(0, true)
    @files.selected_item = @filename_to_control[selected_filename]
    @files.resume_layout
  end

  def populate_files_by_type(file_infos, reg_ex, extension)
    days = @days.value
    now = DateTime.now

    file_infos.each do |info|
      time = now - info.CreationTime
      if time.total_days < days
        match = reg_ex.match(info.name)

        begin
          ticks = System::Int64.Parse(match[2].to_s)
          dt = System::DateTime.FromBinary(0).AddTicks(ticks)

          file_name = match[1] + extension
          file_date = dt
				rescue StandardError, System::Exception
          file_name = info.FullName
          file_date = DateTime.MinValue
        end

        wc = @files.add_item(file_date.to_s, file_name)
        @control_to_filename[wc] = info.FullName
        @filename_to_control[info.FullName] = wc
      end
    end
  end

  def before_close(s, e)
    if(@info.dialog_result == AxDialogResult.OK)
      recover_file
      e.Handled = (@info.dialog_result != AxDialogResult.OK)
    end
  end

  def recover_file
    row =  @files.selected_item
    if (row == nil)
      @info.dialog_result = AxDialogResult.o_k
      return
    end

    file_name = @control_to_filename[row]

    lib_expr = /.*\.rplib$/i
    is_lib = lib_expr.match(file_name) != nil

    save_dialog = Axure::Platform::CommonDialogs::AxSaveFileDialogInfo.new
    save_dialog.filter = !is_lib ? intl_str("Axure RP Files (*.rp)|*.rp") :
            intl_str("Axure RP Library Files (*.rplib)|*.rplib")

    if save_dialog.show_dialog == AxDialogResult.o_k
      new_file_name = save_dialog.file_name

      # add the extension if necessary
      if(is_lib): new_file_name += ".rplib" if !lib_expr.match(new_file_name)
      else new_file_name += ".rp" if !/.*\.rp$/.match(new_file_name)
      end

      error_text = can_use_file(new_file_name)
      if error_text: AxPlatform.platform_services.show_message_box(error_text)
      else System::IO::File.copy(file_name, new_file_name, true)
      end
    else
      @info.dialog_result = AxDialogResult.Cancel
    end
  end

  def can_use_file(file_name)
    file_info = FileInfo.new(file_name)

    # file that doesn't exist is read only
    if file_info.exists
      if file_info.is_read_only: error_text = intl_str("This file is read only. Please save to a different file name.");
      else
        #file exists.  check if we can access it
        begin
          # lock the file
          can_use = AxPlatform.platform_services.lock_document(file_name)
          #release the file
          AxPlatform.platform_services.release_document(file_name)
        rescue StandardError , System::Exception
          can_use = false
        end
      end

      error_text = intl_str "This file is currently in use. Please save using a
different file name or wait until the file becomes available." if !can_use
    end
    error_text
  end
end
