require 'aml'
require 'axure_enumerable'
require 'Model'
require 'interop'

require 'dialogs/package_edit_control.rb'

include Axure::DocumentModel
include Axure::Platform::Controls

class GlobalVariablesEditor < PackageEditControl

  attr_reader :variable_list
  bindable_attr_writer :variable_list

  def initialize()
    #allow AlignPadLayouPanel to size up to its containing control
    self.auto_size = true

    initialize_component("dialogs/global_variables_editor.aml")
  end

  def dialog_info=(info)
    @dialog_info = info

    @dialog_info.default_size = AxSizeF.new(450,320)
    @dialog_info.min_size = AxSizeF.new(300,250)
    @dialog_info.before_close.add(lambda { |sender, e|
      LoseFocus()
      e.Handled = @variable_list.is_editing
    })

    @variables_set = info.properties["GlobalVariableSet"]
    @package_object = @variables_set

    update_list
    @dialog_info.SetProperty("SelectedIndex", @variable_list.SelectedIndex)
  end

  def update_list
    vars = []
    @variables_set.global_variables.each { |gv|
      vars << gv.name
    }

    @variable_list.string_items = vars.to_clr_list if vars.length != 0
  end

  def add_button_clicked
    @variables_set.add_global_variable(next_name(intl_str("NewVariable"), @variables_set.global_variable_names))
    update_list()
    @variable_list.selected_index = @variable_list.get_control_count() - 1
    @variable_list.edit_selected_item()
  end

  def is_valid_variable_name(name)
    return false if @variables_set.GetGlobalVariableByNameCaseInsensitive(name) != nil
    return false if name.length > 25
    return false if name.ToUpper().Equals("CSUM")
    return false if (/^[a-zA-Z0-9]*$/).match(name) == nil
    return false if name.index(" ") != nil
    return false if name.Equals("PageName")

    return true;
  end

  def remove_button_clicked
    if @variables_set.global_variables.count == 1
      AxPlatform.PlatformServices.ShowMessageBox(
        intl_str("At least one variable must exist in the project."))
        return
    end

    if AxPlatform.PlatformServices.ShowMessageBox(
            intl_str("Are you sure you want to delete this variable?\nIt cannot be undone."),
            intl_str("Warning"), AxMessageBoxButtons.YesNo) == AxDialogResult.Yes
      gv = get_selected_gv
      if(gv)
        @variables_set.RemoveGlobalVariable(gv)
        update_list()
        @variable_list.refresh_selection()
      end
    end
  end

  def get_selected_gv
    name = @variable_list.selected_item.text
    return @variables_set.get_global_variable_by_name(name)
  end

  def up_button_clicked
    @variables_set.MoveGlobalVariable(get_selected_gv, -1);
    @variable_list.move_up();
  end

  def down_button_clicked
    @variables_set.MoveGlobalVariable(get_selected_gv, 1);
    @variable_list.move_down();
  end

  def variable_name_edit_finished(s, e)
    new_name = e.control.get_attached_property(ListPanel[].EDITING_TEXT)
    old_name = e.control.text

    return if new_name == old_name

    if(new_name.length > 0 && is_valid_variable_name(new_name))
      get_selected_gv.name = new_name
      e.control.text = new_name
    else
      if AxPlatform.PlatformServices.ShowMessageBox(
              intl_str("The variable name is invalid or in use. Please try again.\nClick Cancel or Press Esc while editing to revert the name"),
              intl_str("Invalid Name"), AxMessageBoxButtons.o_k_cancel) == AxDialogResult.o_k
        #e.control.set_attached_property(ListPanel[].EDITING_TEXT, old_name)
        #@variable_list.edit_selected_item(new_name)
        e.Handled = false
      else
        e.control.set_attached_property(ListPanel[].EDITING_TEXT, old_name)
      end
    end
  end

  def variable_list_selected_index_changed
    @dialog_info.SetProperty("SelectedIndex", @variable_list.SelectedIndex)
    set_remove_button_tooltip()
  end

  def set_remove_button_tooltip
    if (!@variable_list.selected_item || @variable_list.get_control_count > 1): @remove_button.tool_tip = intl_str("Remove")
    else @remove_button.tool_tip = intl_str("At least one variable must exist in the project.")
    end
  end

  #region IPackageEditControl overrides
  def RefreshContent
    @client.CurrentFile.Refresh()
    @variables_set = @client.Document.GlobalVariableSet
    @package_object = @variables_set
    update_list()
  end
  #endregion IPackageEditControl overrides

end
