require 'aml'
require 'dialogs/generators/spec_gen_dlg/general_pane'
require 'dialogs/generators/spec_gen_dlg/pages_pane'
require 'dialogs/generators/spec_gen_dlg/masters_pane'
require 'dialogs/generators/spec_gen_dlg/notes_pane'
require 'dialogs/generators/spec_gen_dlg/screenshot_pane'
require 'dialogs/generators/spec_gen_dlg/annotations_pane'
require 'dialogs/generators/spec_gen_dlg/widgets_pane'
require 'dialogs/generators/spec_gen_dlg/layout_pane'
require 'dialogs/generators/spec_gen_dlg/word_template_pane'

require 'dialogs/package_edit_control.rb'

include System::IO

include Axure::Platform::WindowlessControls
include Axure::Platform::Drawing
include Axure::Client::WindowlessControls
include Axure::Platform

class SpecificationGenerationDialog < PackageEditControl
  attr_reader :configuration, :use_mine_ckbutton
  bindable_attr_writer :configuration, :use_mine_ckbutton

  def configuration=(value)
    raise "configuration is nil" if value == nil

    @configuration = value

    if (@configuration.temp_override_destination_path.length == 0)
      @configuration.temp_override_destination_path = AxPlatform.PlatformServices.GetDefaultSpecificationPath() + SEP + configuration.temp_project_name + ".docx"
    end

    panes.each do |x|
      x.configuration = value
      x.my_init() if x.respond_to?(:my_init)
    end

    @package_object = @configuration
    @dialog_info["specConfig"] = @configuration

    update_binding(:configuration)
  end

  def dialog_info=(value)
    @dialog_info = value
    self.configuration = @dialog_info["specConfig"]
    value.before_shown.add(lambda { @list.focus })

    @dialog_info.before_close.add(lambda { |sender, e|
      e.handled = !validate_before_generate() if @dialog_info.dialog_result == AxDialogResult.OK
      SaveContent()
    })

    @dialog_info.default_size = AxSizeF.new(630,570)
    @dialog_info.min_size = AxSizeF.new(630,570)
  end

  def panes
    @pane_names_to_pane.values
  end

  # returns the pane associated with a name and creates it if necessary
  # this assumes that pane_name is a valid class name
  def get_pane(pane_name)
    pane = @pane_names_to_pane[pane_name]
    if !pane
      pane = Kernel.const_get(pane_name).new
      @pane_names_to_pane[pane_name] = pane
      pane.configuration = configuration
      pane.my_init if pane.respond_to?(:my_init)
    end
    pane
  end

  def initialize()
    @pane_names = ["GeneralPane", "PagesPane", "MastersPane", "NotesPane", "ScreenshotPane", "AnnotationsPane",
                   "LayoutPane", "WordTemplatePane"]
    @pane_names_to_pane = {"GeneralPane" => GeneralPane.new }
    initialize_component('dialogs/generators/spec_gen_dlg/spec_gen_dlg.aml')
    @list.selected_index = 0
  end

  def dispose
    super
    panes.each {|pane| pane.dispose }
  end

  def file_browse_override_button_clicked(sender, args)
    save_dialog = Axure::Platform::CommonDialogs::AxSaveFileDialogInfo.new
    save_dialog.filter = intl_str("Word Documents (*.docx)|*.docx")

    @override_file_text_box.text = save_dialog.file_name if save_dialog.show_dialog == AxDialogResult.o_k
  end

  def reset_override_file_button_clicked
    @override_file_text_box.text = AxPlatform.PlatformServices.GetDefaultSpecificationPath() + SEP + configuration.temp_project_name + ".docx"
  end

  def selected_index_changed(sender, e)
    @group_box.text = sender.selected_item.text
    @group_box.clear_controls()
    x = get_pane(@pane_names[sender.selected_index])
    @group_box.add_control(x)
    x.after_shown() if x.respond_to?(:after_shown)

    @shared_project_box.visible = self.disable_package_edit && @group_box.content.is_a?(GeneralPane)
  end

  #region IPackageEditControl implementation
  def RefreshContent
    oldTempName = @configuration.TempProjectName
    oldOverrideDestinationPath = @configuration.TempOverrideDestinationPath
    oldOverrideDestination = @configuration.TempOverrideDestination
    oldGenerateUsingOverrideDestination = @configuration.TempGenerateUsingOverrideDestination

    newConfig = @client.CurrentFile.LoadPackage(@configuration.PackageId, @configuration.TypeName)
    newConfig.TempOverrideDestinationPath = oldOverrideDestinationPath
    newConfig.TempOverrideDestination = oldOverrideDestination
    newConfig.TempGenerateUsingOverrideDestination = oldGenerateUsingOverrideDestination
    newConfig.TempProjectName = oldTempName

    self.configuration = newConfig
  end

  def disable_package_edit=(value)
    super
    @shared_project_box.visible = self.disable_package_edit
  end

  #endregion IPackageEditControl implementation
  def validate_before_generate()
    if !AxPlatform.platform_services.is_file_closed(@configuration.Word2007SpecOptions.InputDocxFilePath)
      AxPlatform.ShowMessageBox(  intl_str("Please save and close the Input File before generating the specification."))
      return false
    end

    if System::IO::File.Exists(@configuration.GenerateDestinationPath) && !AxPlatform.platform_services.is_file_closed(@configuration.GenerateDestinationPath)
        AxPlatform.ShowMessageBox(  intl_str("The destination file is in use. Please be sure it is not being used by another application before generating the specification."));
        return false
    end

    destinationPath = @configuration.Word2007SpecOptions.OutputDocxFilePath;
    if (@configuration.TempOverrideDestination && self.disable_package_edit)
      destinationPath = @configuration.TempOverrideDestinationPath;
    end

    destinationPath = destinationPath.gsub("/", "\\")
    destinationPath = destinationPath.gsub("\\", SEP)

    ext = ".docx"
    if (destinationPath[-5,5] != ext)
      AxPlatform.ShowMessageBox(intl_str("The destination file name must end with '.docx'. Please specify a new file name."))
      return false
    end

    if !(@configuration.TempOverrideDestination && self.disable_package_edit)
      @configuration.Word2007SpecOptions.OutputDocxFilePath = destinationPath
    else
      @configuration.TempOverrideDestinationPath = destinationPath
    end

    if (destinationPath.length == 0)
      AxPlatform.ShowMessageBox(intl_str("The destination file is not valid.\nPlease select another folder."))
      e.handled = true
      return
    end

    indexofslash = destinationPath.rindex(SEP)
    dir = ""
    if (indexofslash > 0)
      dir = destinationPath[0, indexofslash + 1]
    end

    if (dir.length > 0)
      if (!Directory.Exists(dir))
        if (AxPlatform.ShowMessageBox(  intl_str("The destination folder does not exist.\nWould you like to create it?"),   intl_str("Alert"), AxMessageBoxButtons.YesNo) == AxDialogResult.Yes)
          begin
            System::IO::Directory.CreateDirectory(dir);
					rescue StandardError, System::Exception
            AxPlatform.ShowMessageBox(  intl_str("The destination folder could not be created.\nPlease select another folder."));
            return false
          end
        else
          return false
        end
      end
    else
      AxPlatform.ShowMessageBox(  intl_str("The destination path is not valid.\nPlease try again."));
      return false
    end

    @configuration.TempGenerateUsingOverrideDestination = (@configuration.TempOverrideDestination && self.disable_package_edit)

    return true
  end
end
