require 'aml'
require 'Model'
require 'interop'

include Axure::DocumentModel
include Axure::Client::Generators

class MoreConfigurationsDialog < AML::Control
  attr_accessor :document, :client, :file
  attr_reader :configs
  bindable_attr_writer :configs

  def my_init()
    self.client = @info["client"]
    self.document = client.document
    self.file = client.current_file

    configs.set_column_editable(0, true)
    @info.before_shown.add(lambda { configs.set_widths_proportional })
    reset_configs
  end

  def dialog_info=(value)
    @info = value

    @info.default_size = AxSizeF.new(490,250)
    @info.min_size = AxSizeF.new(420,200)
    my_init()

    @document_changed_lambda = lambda { my_init() if !@disposed }
    self.file.document_changed.add(@document_changed_lambda)
    @info.before_close.add(lambda { |sender, e| before_close(sender, e) })
  end

  def before_close(sender, e)
    #the OK button is just routed to the Edit and Generate button
    if(@info.DialogResult == AxDialogResult.OK)
      e.Handled = true
      open_button_clicked
    end
  end

  def dispose
    super
    @disposed = true
    self.file.document_changed.remove(@document_changed_lambda)
  end

  def initialize()
    initialize_component('dialogs/generators/more_configurations_dlg.aml')
  end

  def reset_configs
    @configs.suspend_layout()
    @configs.clear_controls()
    @item_to_id_hash = Hash.new
    document.generator_config_package_ids.each { |id| add_grid_list_item(id) }
    @configs.refresh_selection()
    update_buttons
    @configs.resume_layout()
  end

  def add_grid_list_item(id)
    info = document.get_package_info(id)
    generator = Generator.get_configuration_type_display_name(info.package_type_name)
    item = @configs.add_item(display_name(id), generator)

    #Set the editing name for the item, which is different from the DisplayName
    item.get_control_at_index(0).set_attached_property(ListPanel[].EDITING_TEXT, info.package_name)
    @item_to_id_hash[item] = id
  end

  def display_name(id)
    info = document.get_package_info(id)
    display_name = info.package_name
    display_name = intl_str1("{0} (default)", display_name) if(is_default(id))
    display_name = intl_str1("{0} (not supported)", display_name) if(info.package_type_name == AxureType.Word2000SpecificationGeneratorConfiguration)
    display_name
  end

  def add_select_list_item_selected(sender, e)
    type_name = Generator.get_configuration_type_name(e.control.text)
    config = Generator.create_generation_configuration(type_name, document)
    id = config.package_id

    document.create_package_info(id, config.type_name)
    config.name = next_name(intl_str("New Configuration"), config_names)

    file.save_package(config)
    document.add_generator_configuration(id)
    add_grid_list_item(id)

    @configs.selected_index = @configs.get_control_count() - 1
    @configs.edit_selected_item()

    @add_selectlist.text = intl_str("Add")
    #p "client.doc = file.document: " + (file.document == document).to_s
  end

  def config_names
    document.generator_config_package_ids.collect do |id|
      document.get_package_info(id).package_name
    end
  end

  def remove_button_clicked
    id = @item_to_id_hash[@configs.selected_item]
    if(is_default(id))
      AxPlatform.platform_services.show_message_box(intl_str("Default configurations cannot be deleted."),
                                                    intl_str("Alert"), AxMessageBoxButtons.o_k)
    else
      document.remove_generator_configuration(id)
      reset_configs

      @configs.refresh_selection
    end
  end

  def open_button_clicked
    edit_config(@item_to_id_hash[@configs.selected_item])
  end

  def edit_config(id)
    should_generate, generator_config = Generator.open_configuration_editor(client,id)
    Generator.generate(file, generator_config, @client) if should_generate
    #todo add a apply or cancel button to sle
    file.save_package(generator_config) if generator_config != nil;
  end

  def duplicate_button_clicked
    info = document.get_package_info(@item_to_id_hash[@configs.selected_item])

    new_info = info.copy(document)
    new_name = next_name(info.package_name, config_names)
    new_id = Guid.new_guid()

    new_info.package_name = new_name
    new_info.package_id = new_id

    document.add_generator_configuration(new_id)
    document.set_package_info(new_info)

    add_grid_list_item(new_id)

    # todo ask martin what this is doing
    dict = Dictionary[Guid, Guid].new
    dict.add(info.package_id, new_info.package_id)
    file.import_packages_from(file, [info].to_clr_list(), dict, Dictionary[Guid, Guid].new)
    #p "duplicateConfigurationButtonClicked client.doc = file.document: " + (file.document == document).to_s
  end

  def default_button_clicked
    info = document.get_package_info(@item_to_id_hash[@configs.selected_item])
    case info.package_type_name
      when AxureType.word2000_specification_generator_configuration, AxureType.word2007_specification_generator_configuration:
         document.default_spec_config_package_id = info.package_id
      when AxureType.html_prototype_generator_configuration: document.default_prototype_config_package_id = info.package_id
    end
    reset_configs
  end

  def configs_edit_finished(sender, e)
    label = e.control
    new_name = label.get_attached_property(ListPanel[].EDITING_TEXT)

    id = @item_to_id_hash[@configs.selected_item]
    if(!id)
      s = "The id is null when finishing an edit in MoreConfigurationsDialog.\n
SelectedItem:#{debug_string_from_item(@configs.selected_item)}\n"

      @item_to_id_hash.each { |item, item_id|
        s += "\nitem:#{debug_string_from_item(item)} id:#{item_id}"
      }
      raise s
    end

    info = document.get_package_info(id)
    gc = file.load_package(info.package_id, info.package_type_name)

    gc.name = new_name
    info.package_name = new_name
    label.text = display_name(id)
  end

  def debug_string_from_item(item)
    if(item)
      if(item.GetControlCount == 0)
        "#{item} Control:Null"
      else
        "#{item} Control:#{item.GetControlAtIndex(0)}"
      end
    else
      "Null"
    end
  end

  def update_buttons
    @info.OKEnabled = @configs.EditButtonEnabled
    item = @configs.selected_item
    if item
      id = @item_to_id_hash[item]

      enabled = !is_default(id)
      @remove_button.enabled = enabled
      if !enabled
        @remove_button.tool_tip = intl_str("Remove: default configurations cannot be deleted.")
      else
        @remove_button.tool_tip = intl_str("Remove")
      end

      # special logic to set the default button enabled and tooltip
      @default_button.enabled = enabled && can_be_default(id)
      if (enabled && !can_be_default(id)):
        @default_button.tool_tip =
                intl_str("Set as default: only Word Specifications and HTML Prototypes be set as default.")
      else
        @default_button.tool_tip = intl_str("Set as default")
      end
    else
      @remove_button.tool_tip = intl_str("Remove")
      @default_button.tool_tip = intl_str("Set as default")
    end
  end

  def configs_selected_index_changed
    update_buttons()
  end

  def configs_item_selected(sender, e)
    e.Handled = true
    edit_config(@item_to_id_hash[e.control])
  end

  def is_default(id)
    id == document.default_prototype_config_package_id || id == document.default_spec_config_package_id
  end

  def can_be_default(id)
    type = document.get_package_info(id).package_type_name
    type == AxureType.word2000_specification_generator_configuration ||
    type == AxureType.word2007_specification_generator_configuration ||
    type == AxureType.html_prototype_generator_configuration
  end
end