

# +CustomObservableBuilder+ is extended by +CustomObservable+ to allow the class that +CustomObservable+ is being
# mixed into to use +observable_attr_writer+
#
#   +observable_attr_writer+ works just like +attr_writer+ except that it calls +notify_observers+ when the property
#   is set.
module CustomObservableBuilder

  # When this module is extended by a class, +observable_attr_writer+ can be used to define a property setter
  # exactly like the traditional +attr_writer+ construct. The defined property setter calls +notify_observers+
  # with the symbol of the set property (the symbol passed is the one for the getter, so <tt>:prop</tt> and not
  # <tt>:prop=</tt>
  def observable_attr_writer(*props)
    props.each do |prop|
      getter = prop.to_sym
      setter = "#{prop}=".to_sym
      var_sym = "@#{getter}".to_sym
      
      self.send(:define_method, setter) do |new_val|
        instance_variable_set(var_sym, new_val)
        notify_observers(getter)
      end
    end
  end
  
end

# +CustomObservable+ provides similar but more fine-grained control than the standard library +Observable+ module.
# Classes that mix in +CustomObservable+ can post notifications by calling the private +notify_observers+ method
# that goes through each of the registered observers and posts the notification to every observer with the appropriate
# and matching pattern.
#
# Observers can be added using the +add_observer+ method which takes a regex, string, or symbol to control which
# events an observer will be notified of
module CustomObservable
  extend CustomObservableBuilder
  
  ObserverInfo = Struct.new(:pattern, :target, :method, :observer_data)

  # Adds an observer to the observer list for a pattern.  When an event matching the pattern is fired, for each observer
  # with an event matching the pattern, +method+ is invoked on the +target+.
  # +observer_data+ allows the observed object to pass in extra state data which will be passed back when the observer
  # is notified.
  # Observer methods can take 0 or more of the following paramaters:
  #   * sender -- the sender of the event
  #   * event -- a symbol representing the event.  For properties, this is the name of the set property
  #   * event_data -- data defined by the event type. This can be discovered via documentation
  #   * observer_data -- the data passed in when the observer is added
  def add_observer(pattern, target, method, observer_data = nil)
    if pattern.instance_of?(Regexp)
      (@regex_observers ||= []) << ObserverInfo.new(pattern, target, method.to_sym, observer_data)
    else
      pattern_sym = pattern.to_sym
      @standard_observers ||= {}
      (@standard_observers[pattern_sym] ||= []) << ObserverInfo.new(pattern_sym, target, method.to_sym, observer_data)
    end
  end

  # Removes observers where both the pattern and target match the passed in parameters
  def remove_observer(pattern, target)
    if pattern.instance_of?(Regexp)
      @regex_observers.delete_if {|x| x.pattern == pattern && x.target == target }
    else
      pattern_sym = pattern.to_sym
      observers_for_pattern = @standard_observers[pattern_sym]
      observers_for_pattern.delete_if {|x| x.target == target }
      @standard_observers.delete(pattern_sym) if observers_for_pattern.length == 0
    end
  end

  # Removes all observers for a specified target
  def remove_observer_target(target)
    @regex_observers.delete_if {|x| x.target == target }
    keys_to_remove = []
    @standard_observers.each_pair do |key, val|
      val.delete_if {|x| x.target == target }
    end
    # remove the empty keys
    @standard_observers.delete_if {|key, val| val.length == 0 }
  end

  protected

  # Notifies all observers that an event has occured.  This can be called with either just the event or the event and
  # data containing information relevant to the event 
  def notify_observers(event, event_data = nil)
    event = event.to_sym
    if infos_to_notify = (@standard_observers || {})[event]
      infos_to_notify.each do |info|
        notify_observer_info(info, event, event_data)
      end
    end

    event_str = event.to_s
    (@regex_observers || []).each do |observer_info|
      notify_observer_info(observer_info, event, event_data) if (observer_info.pattern === event_str)
    end
  end

  private

  # Notifies an individual +ObserverInfo+.  This should only be used by +notify_observers+.
  def notify_observer_info(observer_info, event, event_data = nil)
    method = observer_info.target.method(observer_info.method)
    method_args = [self, event, event_data, observer_info.observer_data]
    if method.arity >= 0
      # call the method with any number of arguments that it takes, up to 3
      method.call(*method_args[0, method.arity])
    else
      # call the method with all the arguments
      method.call(*method_args)
    end
  end

end