# this file contains custom grammars such as
#   %rect{0, 0, 50, 50) for creating a rectangle
#   %sizef{20, 20}
# for creating structs and such simply in AML files

require 'Client'
require 'Platform'

include Axure::Platform::Drawing
include Axure::Platform::WindowlessControls

module AML

end

module AML::Grammars


  # returns a new AxRectangle.
  # A valid value has four parameters separated by spaces and/or commas
  def _aml_rect_(object, prop_name, expr_value)
    vals = get_converted_array(expr_value, 4, 4) {|x| x.to_i }
    set_obj_prop(object, prop_name, AxRectangle.new(*vals))
  end

  # returns a new AxRectangleF.
  # A valid value has four parameters separated by spaces and/or commas
  def _aml_rectf_(object, prop_name, expr_value)
    inv_culture = System::Globalization::CultureInfo.invariant_culture
    vals = get_converted_array(expr_value, 4, 4) {|x| System::Single.parse(x, inv_culture) }
    set_obj_prop(object, prop_name, AxRectangleF.new(*vals))
  end

  # A handy shortcut for %rectf (%r)
  def _aml_r_(object, prop_name, expr_value)
    _aml_rectf_(object, prop_name, expr_value)
  end


  # returns a new AxPoint.
  # A valid value has two parameters separated by spaces and/or commas
  def _aml_point_(object, prop_name, expr_value)
    vals = get_converted_array(expr_value, 2, 2) {|x| x.to_i }
    set_obj_prop(object, prop_name, AxPoint.new(*vals))
  end

  # returns a new AxPointF.
  # A valid value has two parameters separated by spaces and/or commas
  def _aml_pointf_(object, prop_name, expr_value)
    inv_culture = System::Globalization::CultureInfo.invariant_culture
    vals = get_converted_array(expr_value, 2, 2) {|x| System::Single.parse(x, inv_culture) }
    set_obj_prop(object, prop_name, AxPointF.new(*vals))
  end


  # returns a new AxSize.
  # A valid value has two parameters separated by spaces and/or commas
  def _aml_size_(object, prop_name, expr_value)
    vals = get_converted_array(expr_value, 2, 2) {|x| x.to_i }
    set_obj_prop(object, prop_name, AxSize.new(*vals))
  end

  # returns a new AxSizeF.
  # A valid value has two parameters separated by spaces and/or commas
  def _aml_sizef_(object, prop_name, expr_value)
    inv_culture = System::Globalization::CultureInfo.invariant_culture
    vals = get_converted_array(expr_value, 2, 2) {|x| System::Single.parse(x, inv_culture) }
    set_obj_prop(object, prop_name, AxSizeF.new(*vals))
  end


  # allows the setting of grid sizes on a grid layout.  Grid sizes take the form:
  # %grid_sizes{1, 1*, fit, 33*}
  # where the stars indicate proportional sizing of the remaining area, a number indicates a fixed sizing
  # and fit indicating that the row or column should be sized to its contents
  def _aml_grid_sizes_(object, prop_name, expr_value)

    begin
      inv_culture = System::Globalization::CultureInfo.invariant_culture

      to_assign = System::Collections::Generic::List[System::Nullable[GridSize]].new
      vals = expr_value.split(/[,\s]+/)
      vals.each do |val|
        processed = val.gsub(' ', '').downcase
        case
          when processed[-1] == ?*
            # parse the number before the star
            number_text = processed[0, processed.length - 1]
            number = number_text.length == 0 ? 1 : System::Single.parse(number_text, inv_culture)
            to_assign << GridSize.new(GridSizeType.Proportional, number)
          when processed == 'fit'
            to_assign << GridSize.new(GridSizeType.Fit, 0)
          else
            to_assign << GridSize.new(GridSizeType.Fixed, System::Single.parse(processed, inv_culture))
        end
      end
    rescue StandardError => err
      raise "Unable to parse #{expr_value} into a grid sizes"
    end

    set_obj_prop(object, prop_name, to_assign)
  end

  # returns a List<System::String
  # %grid_sizes{1, 1*, fit, 33*}
  # where the stars indicate proportional sizing of the remaining area, a number indicates a fixed sizing
  # and fit indicating that the row or column should be sized to its contents
  def _aml_intl_strings_(object, prop_name, expr_value)
    begin
      to_assign = System::Collections::Generic::List[System::String].new

      vals = eval expr_value
      vals.each do |str|
        to_assign << System::String.new(intl_str(str))
      end
    rescue StandardError => err
      raise "Unable to parse #{expr_value} into a list of strings"
    end

    set_obj_prop(object, prop_name, to_assign)
  end

  def _aml_list_(object, prop_name, expr_value)
    set_obj_prop(object, prop_name, eval(expr_value).to_clr_list)
  end

  private

  # Sets an object property.  Just used to centralize property setting
  def set_obj_prop(object, prop_name, val)
    object.send("#{prop_name}=", val)
  end

  # converts a string value to an array of typed values.
  # the returned typed values are determined by the passed in block
  # Also, min_size and max_size can optionally be passed in to ensure a specific number of values
  def get_converted_array(expr_value, min_size = nil, max_size = nil)
    vals = expr_value.split(/[,\s]+/)
    raise "Error converting #{expr_value} to a rect. Wrong number of values." if max_size && vals.size > max_size
    raise "Error converting #{expr_value} to a rect. Wrong number of values." if min_size && vals.size < min_size
    #inv_culture = System::Globalization::CultureInfo.invariant_culture
    vals.collect! {|x| yield x }
    vals
  end

end
