require 'Core'
require 'Client'
require 'International'
require 'Platform'

require 'xmlreader'
require 'mangle'
require 'char'

require 'aml/aml_grammars'

module AML
  include Axure::Constants
  include Axure::Platform
  include Axure::Platform::Controls
  include Axure::Platform::Drawing
  include Axure::Platform::Theme
  include Axure::Platform::WindowlessControls

  include Axure::Client::Interface
  include Axure::Client::WindowlessControls

  include System
  include System::Xml
  include System::Collections::Generic

  include AML::Grammars

  # this is for convenience so that every ruby function can use this for path separators instead of
  # creating its own
  SEP = System::IO::Path.directory_separator_char

  def AML.get_ruby_control(ruby_control_class, host, info)
    ruby_control = ruby_control_class.new()
    ruby_control.size = host.scroll_view_size if host
    ruby_control.anchor = Position::all
    ruby_control.dialog_info = info if ruby_control.respond_to?(:dialog_info=)

		return ruby_control
	end

#  def AML.create_dialog(dlg_control_class, host, info)
#    dlg_con = get_ruby_control(dlg_control_class, info)
#
#    if info.dialog_mode == Axure::Client::Dialogs::DialogMode.PackageEdit
#      pkg_editor = PackageEditor.new
#      pkg_editor.size = host.scroll_view_size
#      pkg_editor.anchor = Position::all
#
#      pkg_editor.dialog_info = info
#      pkg_editor.client = info.client
#      pkg_editor.content = dlg_con
#
#      host.add_control(pkg_editor)
#    else
#      host.add_control(dlg_con)
#    end
#  end

  # returns the translated string for the passed in string
  def intl_str(text)
    Axure::International::Local.get_string(text)
  end# returns the translated string for the passed in string

  def intl_str1(text, first)
    Axure::International::Local.get_string(text, first)
  end

#  def intl_str2(text, first, second)
#    Axure::International::Local.get_string(text, first, second)
#  end
#
#  def intl_str3(text, first, second, third)
#    Axure::International::Local.get_string(text, first, second, third)
#  end
#
#  def intl_str4(text, first, second, third, fourth)
#    Axure::International::Local.get_string(text, first, second, third, fourth)
#  end

  # a convenience method to show a message box
  def msg_box(text)
    Axure::Platform::AxPlatform.platform_services.show_message_box(text.to_s)
  end

  # this will return you the next available name matching basename(#)
  # ex. list
  # newName1
  # newName
  # newName3
  #
  # calling next_name with "newName" and this list would return "newName2"
  def next_name(base_name, name_array)
#    todo: figure out why this was necessary before, I think it had to do with the regular expression being compiled
#    the first time and then not recompiling the next time it comes through, but now it seems to be working without this hack
#    ians_unique_string = "###ian rules -#777#- google rules###"
    number_array = []
#    expr = /^#{ians_unique_string}(\d*)$/i
    expr = /^#{base_name}(\d*)$/i

    name_array.each do |val|
#      new_string = val.sub(base_name, ians_unique_string)
      new_string = val
      match = expr.match(new_string)
      number_array += match[1] ? [match[1].to_i] : [0] if match
    end

    # this is to make the first item giving back suffixed with a '1'
    number_array += [0]

    number_array.sort!.uniq!
    next_number = 0
    number_array.each do |val|
      if(val != next_number)
        break
      end
      next_number += 1
    end
    if next_number != 0
      base_name + next_number.to_s
    else base_name
    end
  end

  # returns the default name for the AML file associated with a class. When +AML+ is mixed in
  # this will return the class name by default
  def get_default_aml_name()
    self.class.name
  end

  # retrievs the actual class name of a class without the namespace:
  # (e.g.) where <tt>cls.name</tt> would be <tt>Axure::Platform::WindowlessControls::WindowlessControl</tt>
  # this method would return +WindowlessControl+
  def get_class_name_without_namespace(cls)
    cls_name = cls.name
    last_dbl_colon = cls_name.rindex('::')
    start_index = last_dbl_colon ? last_dbl_colon + 2 : 0
    cls_name[start_index, cls_name.length]
  end

  # for each directory in the load path, attempts to load the aml file in this order:
  #   manle_name(aml_name), aml_name.downcase, aml_name
  # for example, for TestName this method would attempt to load:
  #   test_name.aml, testname.aml and TestName.aml on each directory in the file
  def get_aml_file_to_load(aml_name)
    mangled_name = Mangle.mangle_name(aml_name)
    downcased_name = aml_name.downcase
    $:.reverse.each do |path|
      to_test = [
        "#{path}/#{mangled_name}.aml",
        "#{path}/#{downcased_name}.aml",
        "#{path}/#{aml_name}.aml",
        "#{path}/#{aml_name}"] # the last one is with no extension

      to_test.each do |current_file|
        return current_file if File.exists?(current_file)
      end
    end
    # if we get here, we failed to find anything
    return nil;
  end


  def skip_to_next_element_tag(xml_reader)
    until xml_reader.node_type == XmlNodeType::element || xml_reader.node_type == XmlNodeType::end_element
      more_nodes = xml_reader.read
      return nil if !more_nodes
    end
  end

  def set_attribute_properties(current_object, reader)
    reader.attribute_count.times do
      reader.move_to_next_attribute

      attr_name = reader.name.to_s
      attr_val = reader.value.to_s
      if attr_name[0, 4] == "aml."
        set_custom_aml_property(current_object, attr_name, attr_val)
      else
        set_property_to_string(current_object, attr_name, attr_val)
      end
    end
  end

  # checks if the property on an object is of type IEnumerable and constructs the appropriate List[T]
  # to assign to that property.  This method returns null if the property is NOT of type IEnumerable
  def get_list_for_properties(current_object, prop_tag_property)
    obj_type = current_object.GetType()
    prop_info = obj_type.get_property(prop_tag_property)
    raise "couldn't find property: #{prop_tag_property}" if !prop_info
    prop_type = prop_info.property_type
    prop_list = nil
    if prop_type.get_interface("IEnumerable")
      #prop_list = []
      type_args = prop_type.get_generic_arguments
      generic_arg = type_args.get_value(0) # bug with indexing arrays of System::Type
      if generic_arg
        types = System::Array[System::Type].new(1)
        types[0] = generic_arg
        list_type = System::Collections::Generic::List.to_clr_type
        list_generic_type = list_type.make_generic_type(types)
        prop_list = System::Activator.create_instance(list_generic_type)
      else
        prop_list = System::Collections::Generic::List[System::Object].new
      end
    end
    prop_list
  end

  def read_and_set_property_element (current_object, reader)
    property_element_name = reader.name.to_s
    prop_tag_property = get_property_tag_property(reader.name.to_s)

    prop_list = get_list_for_properties(current_object, prop_tag_property)
    sub_obj = nil

    reader.read
    while !(reader.name == property_element_name && reader.node_type == XmlNodeType::end_element)
      skip_to_next_element_tag(reader)
      # this MUST be an object node
      sub_obj = process_element(reader)
      prop_list.add(sub_obj) if prop_list

      # the reader is should now be at the end element of the contained object. we need to read until the next
      # element and max sure it's the end attached property element
      reader.read
      skip_to_next_element_tag(reader)
    end

#    if !(reader.name == property_element_name && reader.node_type == XmlNodeType::end_element)
#      raise "Cannot contain more than one object inside an atttached property"
#    end

    if prop_list
      current_object.send("#{prop_tag_property}=", prop_list)
    else
      current_object.send("#{prop_tag_property}=", sub_obj)
    end


  end

  def get_property_tag_property(tag_name)
    tag_name[tag_name.index('.') + 1, tag_name.length]
  end

  def add_sub_object_to_current(current_object, sub_obj)
    return current_object.add_control(sub_obj) if current_object.respond_to?(:add_control)
    return current_object.add(sub_obj) if current_object.respond_to?(:add)
    return current_object.push(sub_obj) if current_object.respond_to?(:push)

    raise "Don't know what to do with sub object of type #{sub_obj.class.name}, #{current_object.class.name} doesn't respond to :add_control, :add or :push"
  end

  def is_property_tag_for_parent? (parent_node_name, sub_node_name)
    return sub_node_name[0, 2] == '_.' ||
            sub_node_name[0, parent_node_name.length + 1] == "#{parent_node_name}."
  end


  def process_element(reader)
    # when this is called, we are an an element, and it's NOT an property tag
    node_name = reader.name.to_s

    current_obj_class = self.class.const_get(reader.name)
    raise "Could not find a class named #{reader.name}, are you sure you've imported the proper modules?" if !current_obj_class

    # starts the creation of a new element
    current_object = current_obj_class.new

    # need to do this here because we're changing the state of the reader below
    is_empty_element = reader.is_empty_element
    #read and set all the attributes
    if (reader.has_attributes)
      set_attribute_properties(current_object, reader)
      # gotta make sure we read, because we're still on the old element
    end

    # ok... now, if it's an empty element, we're done reading and we can just return the current object
    # an empty element is a node of the form <abc prop="val" />
    return current_object if is_empty_element

    # if we got here, we want to skip to the next element (start or end) tag
    reader.read
    skip_to_next_element_tag(reader)
    until reader.name == node_name && reader.node_type == XmlNodeType::end_element
      #skip_to_next_element_tag(reader)
      # this checks whether the child node is of the form Parent.Property
      if is_property_tag_for_parent?(node_name, reader.name.to_s)
        read_and_set_property_element(current_object, reader)
        # we need to read one more, because when that method returns, the current node is the close tag of the
        # attached property
        reader.read
      else
        sub_obj = process_element(reader)
        add_sub_object_to_current(current_object, sub_obj)
        # we need to read one more, because when that method returns, the current node is the close tag of the
        # sub object
        reader.read
      end
      # look for the end tag
      skip_to_next_element_tag(reader)
    end

    # when we finally get here, we've got a proper closing tag and we can just return current_object
    current_object
  end


  def load_aml(reader)
    skip_to_next_element_tag(reader)
    while !reader.e_o_f do
      con = process_element(reader)
      # we need to make sure to read, because this is still pointing at the end element or the empty element
      reader.read
      add_sub_object_to_current(self, con)
      skip_to_next_element_tag(reader)
    end
  end


  # sets a property that is an event on an object
  # this method depends an the fact that all event handlers have two and only two
  # arguments
  def set_event_handler(object, obj_type, event_info, prop_name, prop_str_value)
    # if event info is nil, this isn't an event and that means there's no property for this guy
    raise(StandardError, "Could not find property #{prop_name} on #{obj_type}") if !event_info

    handler = method(prop_str_value)
    #if handler is null, we couldnt find the property name
    raise(StandardError, "Could not find method #{prp_str_value} for property #{prop_name}") if !handler

    case handler.arity
    when 0: object.send(prop_name) { send(prop_str_value) }
    when 1: object.send(prop_name) { |x, y| send(prop_str_value, x) }
    when 2: object.send(prop_name) { |x, y| send(prop_str_value, x, y) }
    end


  end

  # this gets the value of a property from a string which would be evaluated with self as root
  # eg, an example would be a +property_string+ "abc" would be evaluated as root.send("abc")
  # (but is much much faster than eval)
  #   property_string can be in the format of: "Const::OtherConst.property.another_property"
  #
  # One important feature/limitation of note is that if the first letter is capitalized, the item
  # before the dot is interpreted as a constant
  # This matters only when the property is a .NET non-mangled property (eg: self.Stuff) where self
  # was omitted.  To avoid this, merely prepend "self" or use mangled names
  def AML.get_property_value(root, property_string)
    props = property_string.split('.')
    start_index = 0
    #current_object = nil

    # if the first character of the first property is a constant, we need to get the constant
    if (?A .. ?Z) === props[0][0]
      constants = props[0].split("::")
      current_context = root.class
      constants.each do |const|
        current_context = current_context.const_get(const)
        raise(StandardError, "Couldn't interpret '#{property_string}") if !current_context
      end

      start_index = 1
    else
      current_context = root
    end

    # after we've got the constant out of the way ("Constant.Constant" is not allowed)
    props[start_index .. props.length - 1].each do |prop|
      current_context = current_context.send(prop)
      return nil if !current_context
    end

    current_context
  end


  # Converts the lat part of +prop_name+ to uppercase, so <tt>Abc.blah</tt> becomes <tt>Abc.BLAH<tt>
  # and returns the attached property
  def get_attached_property_prop(prop_name)
    dot_index = prop_name.rindex('.')
    attached_prop_name = "#{prop_name[0 , dot_index]}.#{prop_name[dot_index+1, prop_name.length].upcase}"

    AML.get_property_value(self, attached_prop_name)
  end


  def AML.does_property_affect_property_string?(property, property_string)
    props = property_string.split('.')

    # if the first character of the first property is a constant, we need to get the constant
    return false if (?A .. ?Z) === props[0][0]
    first_prop = props[0]
    first_prop = Mangle.mangle_name(first_prop) if !Mangle.is_mangled?(first_prop)
    return (first_prop == property.to_s)
  end

  # Binds two properties together so that when the value of <tt>obj.prop_name</tt> changes, so does the
  # object value indicated with the binding expression
  # The value of everything befor the final dot (eg the object that is being bound to), is evaluated using
  # +get_property_value+.
  # Bindings carry some overhead as events handlers will be invoked whenever any property changes. Also, during
  # initialization, <tt>%static_bind</tt> is faster because <tt>%bind</tt> needs to inspect the objects for
  # event handlers.
  def _aml_bind_(object, prop_name, expr_value)
    @binder.add_binding_expr(object, prop_name, expr_value, true, true)
  end

  # Binds the passed in prop to the current variable, this will then set this variable when ever the passed in
  # property OnChanged event is fired.
  def _aml_bind_to_me_(object, prop_name, expr_value)
    @binder.add_binding_expr(object, prop_name, expr_value, true, false)
  end

  # Binds the current variable to the passed in prop, this will then set the passed in property whenever this
  # variables OnChanged event is fired.
  def _aml_bind_from_me_(object, prop_name, expr_value)
    @binder.add_binding_expr(object, prop_name, expr_value, false, true)
  end

  # evaluates the property of using +get_property_value+ ond the resulting value is sent to the <tt>prop_name=</tt>
  # method on object
  # <tt>%static_bind</tt> differs from <tt>%bind</tt> in that the expression is evaluated once, and when the property or the target value
  # changes no attempts are made to keep the properties in sync.
  # Also <tt>%static_bind</tt> carries virtually no overhead for using it (at creation time or after)
  def _aml_static_bind_(object, prop_name, expr_value)
    object.send("#{prop_name}=", AML.get_property_value(self, expr_value))
  end

  # substitutes in the internationalized string.  Does not provied any additional conversions (eg to ints ore enums),
  # so this expression type should only be used for text values
  def _aml_intl_(object, prop_name, expr_value)
    set_property_to_string_val(object, prop_name, intl_str(expr_value))
  end

  # evaluates
  def _aml_eval_(object, prop_name, expr_value)
    object.send("#{prop_name}=", eval(expr_value))
  end


  def update_binding(property)
    @binder.update_binding(property)
  end


  def set_custom_aml_property (object, prop_name, prop_val)
    aml_prop = prop_name[4,prop_name.length].downcase
    if aml_prop == "name"
      if self.respond_to?("#{prop_val.strip}=")
        self.send("#{prop_val.strip}=", object)
      else instance_variable_set("@#{prop_val.strip}", object)
      end
    end
  end

  # Attribute strings can be interpreted two different ways, either they can be interpreted as expressions (if of the form
  # <tt>%expr{expression}</tt>) or as a literal value. This function decides which one to use.
  def set_property_to_string(object, prop_name, prop_str_value)
    if prop_str_value[0] == ?%
      set_property_to_string_expr(object, prop_name, prop_str_value)
    else
      set_property_to_string_val(object, prop_name, prop_str_value)
    end
  end

  # For AML attributes set to a value of the form <tt>%text{expr}</tt> attribute, this
  # property invokes the proper expression handler.  These can be augmented merely by defining a
  # method named +_aml_expr_+ (the two underscores are used to avoid name collisions)
  def set_property_to_string_expr(object, prop_name, prop_str_value)
    percent_index = prop_str_value.index('%')
    first_curly_index = prop_str_value.index('{')
    # this is "eval", "bind", "sbind", "intl"
    expr_cmd = prop_str_value[percent_index + 1 .. first_curly_index - 1].strip
    last_curly_index = prop_str_value.rindex('}')
    expr_value = prop_str_value[first_curly_index + 1 .. last_curly_index - 1]

    method_name = "_aml_#{expr_cmd}_"
    raise(StandardError, "Unknown expression type #{expr_cmd}, no corresponding #{method_name} method found.") if !self.respond_to?(method_name)

    is_attached_prop = prop_name.index(".")

    if !is_attached_prop
      self.send(method_name, object, prop_name, expr_value)
    else
      #aml_set_attached_property(AML.get_property_value(self, prop_name), expr_value)
    end

  end

  # Sets a property on an object using a string value.  this assumes that the object is a CLR object
  # and that a property exists on there.  The matching semantics occur in this order
  #    * if the property is an event, an event handler is attached to a function with that string name
  #   * if the property is a string, that property is assigned
  #    * if the property is an enum, then that enum is parsed
  #    * if the property type supports the <Type>.Parse(string, IFormatProvider) that is used to convert the prop
  #    * if the property supports the <Type>.Parse(string), that property is used
  def set_property_to_string_val(object, prop_name, prop_str_value)
    obj_type = object.GetType()
    prop_info = obj_type.get_property(prop_name)

    if !prop_info
      event_info = obj_type.GetEvent(prop_name)
      if event_info
        # if we get here, it's either an event or an error
        set_event_handler(object, obj_type, event_info, prop_name, prop_str_value)
        return
      end
    end

    # stuff for attached properties
    attached_prop_index = prop_name.index(".")
    if attached_prop_index
      attached_prop = get_attached_property_prop(prop_name)
      prop_type = attached_prop.property_type
    else
      raise(StandardError, "#{prop_name} is not a valid property") if !prop_info
      prop_type = prop_info.property_type
    end

    # we still need to attempt to set the attached props
    if prop_type == System::String.to_clr_type
      prop_val = prop_str_value.to_clr_string
    elsif prop_type.is_subclass_of(System::Enum.to_clr_type)
      prop_val = System::Enum.Parse(prop_type, prop_str_value, true)
    else
      parse_method = prop_type.get_method("Parse",
                                          System::Array.of(System::Type).new(
                                                  [System::String.to_clr_type, System::IFormatProvider.to_clr_type]))
      if parse_method
        prop_val = parse_method.invoke(nil,
                                       System::Array.of(System::Object).new([prop_str_value.to_clr_string,
                                                                             System::Globalization::CultureInfo.invariant_culture]))
      else
        parse_method = prop_type.get_method("Parse",
                                            System::Array.of(System::Type).new([System::String.to_clr_type]))
        if parse_method
          prop_val = parse_method.invoke(nil,
                                         System::Array.of(System::Object).new([prop_str_value.to_clr_string]))
        else
          raise "Don't know how to set #{prop_str_val} to type of #{obj_type}"
        end
      end
    end


    if !attached_prop_index
      object.set_property(prop_name, prop_val)
    else
      aml_set_attached_property(object, attached_prop, prop_val)
    end
  end

  def aml_set_attached_property(object, attached_prop, prop_val)
    object.set_attached_property(attached_prop, prop_val)
  end

  def initialize_component(aml_name = nil)
    aml_name = get_default_aml_name() if !aml_name
    file_to_load = get_aml_file_to_load(aml_name)
    raise(StandardError, "Could not find an AML file to load: '" + aml_name + "'.") if !file_to_load

    # create the binder
    @binder = AML::Binder.new(self)

    XmlReader.load_file(file_to_load) do |reader|
      begin
        load_aml(reader)
      rescue StandardError => err
        raise StandardError, "Error parsing AML #{aml_name} at line: #{reader.line_number}, pos: #{reader.line_position} \n#{err.message}", err.backtrace
      end
    end

  end

end

class AML::SingleBinding
  include System

  attr_reader :sending_obj, :first_property
  attr_reader :weak_listening_obj, :second_property

  def initialize(sending_obj, sending_property, sending_event,
          listening_obj, listening_property_getter, listening_property_setter)
    @sending_obj, @weak_listening_obj = sending_obj, WeakReference.new(listening_obj)
    @sending_event = sending_event.to_sym
    @sending_property = sending_property.to_sym
    @listening_property_getter = listening_property_getter.to_sym
    @listening_property_setter = listening_property_setter.to_sym

    add_listener()
  end

  # We need to handle this in as separate function because the initialize has +listening_obj+ as part of
  # its execution environment.  This means that a lambda defined there could cause problems because it would
  # keep a strang reference to +listening_obj+
  def add_listener()
    @event_proc = proc { handle_event }
    @sending_obj.send(@sending_event).add(@event_proc)
  end

  def handle_event()
    if !@weak_listening_obj.is_alive
      @sending_obj.send(@sending_event).remove(@event_proc) if @event_proc
      @event_proc = nil
    else
      listening_obj = @weak_listening_obj.target
      new_val = @sending_obj.send(@sending_property)
      if listening_obj.send(@listening_property_getter) != new_val
        listening_obj.send(@listening_property_setter, new_val)
      end
    end
  end

  def cleanup_event()
    @sending_obj.send(@sending_event).remove(@event_proc) if @event_proc
    @event_proc = nil
  end
end

class AML::MultiBinding
  include System

  attr_reader :sending_obj
  attr_reader :sending_prop_to_listener_infos

  ListenerInfo = Struct.new(:weak_listener, :prop_getter, :prop_setter, :binding_expr)

  def initialize(sending_obj)
    @sending_obj = sending_obj
    @event_proc = proc {|sender, args| handle_event(sender, args) }
    @sending_obj.property_changed.add(@event_proc)

    @sending_prop_to_listener_infos = Hash.new
  end

  def handle_event(sender, args)
    prop_sym = Mangle.mangle_name(args.property_name).to_sym
    if pairs = @sending_prop_to_listener_infos[prop_sym]
      cleanup_listener_infos(pairs)
      if !pairs.empty?
        set_listener_properties(pairs, prop_sym)
      end
    end
  end

  def cleanup_event
    @sending_obj.property_changed.remove(@event_proc) if @event_proc
    @event_proc = nil
  end

  def add_listener(sending_property, listener_obj, listner_property_getter, listner_property_setter, binding_expr)
    sending_property = sending_property.to_sym
    (@sending_prop_to_listener_infos[sending_property] ||= []) <<
            ListenerInfo.new(WeakReference.new(listener_obj), listner_property_getter.to_sym,
                             listner_property_setter.to_sym, binding_expr)
  end

  def remove_listener(binding_expr)
    keys_to_remove = []
    @sending_prop_to_listener_infos.each do |key, value|
      listeners = value
      listeners.delete_if do |l|
        l.binding_expr == binding_expr
      end

      keys_to_remove << key if listeners.length == 0
    end

    keys_to_remove.each {|key| @sending_prop_to_listener_infos.delete(key)}
  end

  def has_listeners?
    return @sending_prop_to_listener_infos.length > 0
  end

  private

  def cleanup_listener_infos(listener_infos)
    listener_infos.delete_if do |info|
      weak_listener = info.weak_listener
      !weak_listener.is_alive
    end
  end

  def set_listener_properties(listener_prop_pairs_list, sending_prop_sym)
    # it's theoretically possible that in between the cleanup and the setting of listener properties
    # a weak ref went out of scope, so we should make sure that the listener_obj is non-null
    listener_prop_pairs_list.each do |info|
      listener_obj, getter, setter = info.weak_listener.target, info.prop_getter, info.prop_setter
      if listener_obj && listener_obj.send(getter) != @sending_obj.send(sending_prop_sym)
        # listener_prop is a setter
        listener_obj.send(setter, @sending_obj.send(sending_prop_sym))
      end
    end
  end

end

class AML::Binder

  BindingExpression = Struct.new(:object, :prop_name, :expr_value, :bind_to_me, :bind_from_me)

  def initialize(binding_root)
    @binding_root = binding_root
    @binding_expressions_to_bindings = {}

    @single_bindings = []
    # this is for "PropertyChanged" events
    @obj_to_multi_binding = Hash.new
  end

  def add_binding_expr(object, prop_name, expr_value, bind_to_me, bind_from_me)
    binding_expr = BindingExpression.new(object, prop_name, expr_value, bind_to_me, bind_from_me)
    do_bind(binding_expr)
  end

  def do_bind(expr)
    created_bindings = []
    bind_to_object, bind_to_property = get_bind_to_obj_and_property(expr.expr_value)

    if bind_to_object
      expr.object.send("#{expr.prop_name}=", bind_to_object.send(bind_to_property))

      # set up bindings both ways
      created_bindings.concat one_way_bind(expr.object, expr.prop_name, bind_to_object, bind_to_property, expr) if expr.bind_to_me
      created_bindings.concat one_way_bind(bind_to_object, bind_to_property, expr.object, expr.prop_name, expr) if expr.bind_from_me
    end

    @binding_expressions_to_bindings[expr] = created_bindings
  end

  def update_binding(property)
    to_rebind = [] # we need to collect them for later because we'll be modifying the collection
    @binding_expressions_to_bindings.each_key do |expr|
      if (AML.does_property_affect_property_string?(property, expr.expr_value))
        to_rebind << expr
      end
    end

    to_rebind.each do |expr|
      remove_binding_expression(expr)
      do_bind(expr)
    end
  end

  def clear_bindings
    return if @binding_expressions_to_bindings == nil
    to_clear = []
    @binding_expressions_to_bindings.each_key { |expr| to_clear << expr }
    to_clear.each do |expr|
      remove_binding_expression(expr)
    end
  end

  # Binds a source <tt>[object, property]</tt> pair to a target <tt>[object, property]</tt> pair
  # This creates a two-way binding so when either property changed, the corresponding property
  # on the other is changed
  #
  # This method returns the bindings that were created
  def one_way_bind(target_obj, target_property, source_obj, source_property, binding_expr)
    return_val = []
    source_property = Mangle.mangle_name(source_property) if !Mangle.is_mangled?(source_property)
    target_property = Mangle.mangle_name(target_property) if !Mangle.is_mangled?(target_property)

    source_event_name = "#{source_property}_changed"
#    target_event_name = "#{target_property}_changed"

    source_property_sym = source_property.to_sym
    target_property_sym = target_property.to_sym

#    source_property_setter_sym = "#{source_property}=".to_sym
    target_property_setter_sym = "#{target_property}=".to_sym

    # so here we bind the source property to the target property.  The first case takes care of the case
    # when it's a "BlahChanged" event, and the second case takes care of the PropertyChanged where the property
    # raises the PropertyChanged event
    # we only want to bind the event handlers if getting and setting will succeed
    if (source_obj.respond_to?(source_event_name) && target_obj.respond_to?(target_property_setter_sym))
      new_binding =  AML::SingleBinding.new(source_obj, source_property_sym, source_event_name, target_obj,
                                            target_property_sym, target_property_setter_sym)
      @single_bindings << new_binding
      return_val << new_binding
    elsif source_obj.respond_to?(:property_changed)
      # get the multi binding and create it if needed
      multi_binding = (@obj_to_multi_binding[source_obj] ||= AML::MultiBinding.new(source_obj))
      multi_binding.add_listener(source_property, target_obj, target_property_sym, target_property_setter_sym, binding_expr)
      return_val << multi_binding
    end

#    # here is where we do a reverse binding (eg, target to source) this is exactly the same as obove
#    if (target_obj.respond_to?(target_event_name) && source_obj.respond_to?(source_property_setter_sym))
#      new_binding = AML::SingleBinding.new(target_obj, target_property_sym, target_event_name, source_obj,
#                                            source_property_sym, source_property_setter_sym)
#      @single_bindings << new_binding
#      return_val << new_binding
#    elsif target_obj.respond_to?(:property_changed)
#      # get the multi binding and create it if needed
#      multi_binding = (@obj_to_multi_binding[target_obj] ||= AML::MultiBinding.new(target_obj))
#      multi_binding.add_listener(target_property, source_obj, source_property_sym, source_property_setter_sym, binding_expr)
#      return_val << multi_binding
#    end

    return_val
  end

  private

  def get_bind_to_obj_and_property(expr_value)
    last_dot = expr_value.rindex('.')
    if !last_dot
      bind_to_object = @binding_root
      bind_to_property = expr_value
    else
      bind_to_object = AML.get_property_value(@binding_root, expr_value[0 .. last_dot - 1])
      bind_to_property = expr_value[last_dot +  1 .. expr_value.length]
    end

    [bind_to_object, bind_to_property]
  end

  def remove_binding_expression(binding_expr)
    to_remove = @binding_expressions_to_bindings[binding_expr] || []
    to_remove.each do |binding|
      #wow this is weird, the clean up process for IronRuby can get a little funky and delete my class before
      #my resources have been disposed, really we shouldn't be disposing dialogs on close of the app (they should
      #already be disposed) however if for whatever reason they are held onto then our class, AML::SingleBinding
      #is no longer guaranteed to be around, it seems like this is definitely a bug in IronRuby, but hard to define

      #Here, I am using the fact that AML::SingleBinding is no longer around to let me know that I am in a bad and
      #that the app is closing down, so I can just exit this function without doing the normal cleanup
      return if !AML::SingleBinding

      if binding.is_a?(AML::SingleBinding)
        @single_bindings.delete(binding)
        binding.cleanup_event
      else # multi-binding
        binding.remove_listener(binding_expr)
        if !binding.has_listeners?
          @obj_to_multi_binding.delete(binding.sending_obj)
          binding.cleanup_event
        end
      end
    end
    @binding_expressions_to_bindings.delete(binding_expr)
  end

end


module AML::Bindable

  def bindable_attr_writer(*props)
    props.each do |prop|
      getter = prop.to_sym
      setter = "#{prop}=".to_sym
      var_sym = "@#{getter}".to_sym

      self.send(:define_method, setter) do |new_val|
        old_val = instance_variable_get(var_sym)
        if(!old_val.equal?(new_val))
          instance_variable_set(var_sym, new_val)
          update_binding(getter)
#          if(on_changed)
#            on_changed.arity == 1 ? on_changed.call(old_val) : on_changed.call()
#          end
        end
      end
    end
  end

  def onchange_attr(*props)
    props.each do |prop|
      getter = prop.to_sym
      setter = "#{prop}=".to_sym
      var_sym = "@#{getter}".to_sym

      event_sym =  "@#{prop}_changed".to_sym
      event_sym_setter =  "#{prop}_changed=".to_sym
      invoke_event_sym =  "on_#{prop}_changed".to_sym

      self.send(:define_method, event_sym_setter) do |val|
        instance_variable_set(event_sym, val)
      end

      self.send(:define_method, invoke_event_sym) do
        a = instance_variable_get(event_sym)
        a.call() if a
      end

      self.send(:define_method, setter) do |new_val|
        old_val = instance_variable_get(var_sym)
        if(!old_val.equal?(new_val))
          instance_variable_set(var_sym, new_val)
          self.send(invoke_event_sym)
        end
      end
    end
  end

#  def bindable_attr_writer(*props, &on_changed)
#    p "!!#{self.name}!!"
#
#    to_eval = ""
#    props.each do |prop|
#      to_eval << "
#        #class #{self.name}
#          def #{prop}=(new_val)
#            instance_variable_set(:@#{prop}, new_val)
#            update_binding(:#{prop})
#          end
#        #end
#      "
#    end
#
#    p "EVAL"
#    p to_eval
#    p "END EVAL"
#
#    eval to_eval
#  end
end

class RubyDialogInfo
  def method_missing(name, *args, &block)
    name[-1] == ?= ? set_property(name, args[0]) : get_property(name)
  end

  def my_test_method()
    p :the_test_worked
  end
end

class AML::Control < Axure::Platform::WindowlessControls::AmlControlBase
  include AML
  extend AML::Bindable

#  def dispose_children(control)
#    control.controls.each do |child|
#      dispose_children(child)
#      child.dispose if child.is_a?(AML::Control)
#    end
#  end

  def dispose
#    puts "disposing type: #{self.class.name}"
    dialog_info = nil if respond_to?(:dialog_info=)
    @binder.clear_bindings if @binder
#    dispose_children(self)
    super
  end
end
